/*****************************************************************************/
/* File         srchstr.c                                                    */
/*                                                                           */
/* Purpose      Search a string for another                                  */
/*                                                                           */
/* Tested       26 APR 90       OS/2    IBM C2 1.1                           */
/*              26 APR 90       VM      C/370                                */
/*               1 JUN 90       VM      WS                                   */
/*               4 APR 91       AIX     RT                                   */
/*              12 APR 91       AIX     S/6000                               */
/*              11 JUL 91       OS/2 2  CL386                                */
/*                                                                           */
/* Author       Joe Kesselman (KESSELMN at KGNVMF)                           */
/*              Tim Bell      (BELLTG at WINVMB)                             */
/*              Kurt Shoens   (SHOENS at ALMADEN)                            */
/*                                                                           */
/* History                                                                   */
/*              10 NOV 89       Created                                      */
/*               1 JUN 90       Added case insensitive search                */
/*                              Changed interface to support signed chars    */
/*              10 OCT 90       Bug fix if last character is upper case      */
/*                                in strsrhi                                 */
/*              19 OCT 90       Added additional entry points for multiple   */
/*                                searches                                   */
/*                              Added delta2 jump code for partial matches   */
/*               6 NOV 90       Added DELTA2_SKIP flag to disable delta2     */
/*                                code                                       */
/*              27 MAR 91       Zero length pattern caused exception on      */
/*                                S/6000.                                    */
/*               4 APR 91       Added SHORT_SEARCH for those who do not      */
/*                                need the overhead of BM searching          */
/*              12 APR 91       Improve portability for case ignore          */
/*                                searching on a zero length string          */
/*              18 APR 91       Added alternative fast search algorithm      */
/*               7 MAY 91       Strncmpi should be included for S/6000       */
/*              11 JUL 91       Changed all references to strncmpi to        */
/*                               strnicmp for OS/2 2.0 support               */
/*****************************************************************************/
/*****************************************************************************/
/* A fast string searching Algorithm                                         */
/*                                                                           */
/* R.S. Boyer, J. Strother Moore (CACM, Vol. 20 No. 10 (1977 P762-772)       */
/*                                                                           */
/* This algorithm is enabled by compiling with the macros SHORT_SEARCH       */
/* and RADIX_SEARCH set to 0.  This is the default, so you can just compile  */
/* without setting any macros on the command line                            */
/*                                                                           */
/* A string searching routine has two inputs                                 */
/*                                                                           */
/* 1) A string                                                               */
/* 2) A pattern                                                              */
/*                                                                           */
/* It is assumed that the string is long, in the sense that it is worthwhile */
/* to perform some preprocessing on the pattern.  Pre-processing is only     */
/* performed on the pattern, so the preparation is independent of the length */
/* of the string to search.                                                  */
/*                                                                           */
/* The match will take place from right to left in the string.  There are    */
/* two tactics we can use                                                    */
/*                                                                           */
/* 1) Consider that we are trying to find the string "INGROWING" in          */
/*    the string "TEST MATCH HERE"                                           */
/*                                                                           */
/*    TEST MATCH HERE                                                        */
/*    INGROWING                                                              */
/*                                                                           */
/*    We first compare the final character at the end of "INGROWING", i.e.   */
/*    the 'G'.  It does not match, as 'C' is not equal to 'G'.               */
/*    In the simple algorithm, we would just move forward one character and  */
/*    compare the 'G' to the 'H'.  However, we can actually move 9           */
/*    characters forward, but using the information that 'C' is not in the   */
/*    string "INGROWING" at all!  Thus, we move to                           */
/*                                                                           */
/*    TEST MATCH HERE                                                        */
/*             INGROWING                                                     */
/*                                                                           */
/*    This cannot match, so we have succeeded, with one character compare    */
/*    detecting that there is no match!                                      */
/*                                                                           */
/*    There is a cost, though.  In order to determine whether there was a 'C'*/
/*    in the pattern, we had to process it.  We build a table of characters  */
/*    that failed to match to offsets to move the string.  For the chacters  */
/*    that are not in the pattern, that offset is the length of the pattern. */
/*    For those that are in the pattern, it is the offset to align the       */
/*    character with the right most character in the pattern.                */
/*                                                                           */
/*    For example,                                                           */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*    INGROWING                                                              */
/*                                                                           */
/*    We test that the 'I' does not match the 'C' and shift the pattern till */
/*    the 'I' in the string is aligned with the 'I' in the pattern.          */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*      INGROWING                                                            */
/*                                                                           */
/*    We now repeat the search ... as we started comparing at the last       */
/*    character, we move backwards towards the start,                        */
/*    'G' matches 'G', 'N' matches 'N', 'I' matches 'I', 'W' does not match  */
/*    'R'.  So we shift the pattern again to align the 'R'                   */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*        INGROWING                                                          */
/*                                                                           */
/*    'G' does not match 'S', so                                             */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*                 INGROWING                                                 */
/*    ' ' does not match 'G', so we end the search.                          */
/*                                                                           */
/*    So, the algorithm to generate the jump table is as follows             */
/*                                                                           */
/*    For all characters in the alphabet                                     */
/*      DELTA(char)=pattern length                                           */
/*    For I = 1 to pattern length                                            */
/*      DELTA(pattern(i)) = pattern length - i                               */
/*                                                                           */
/*    Thus, for "INGROWING", that generates                                  */
/*                                                                           */
/*      DELTA('G') = 0                                                       */
/*      DELTA('N') = 1                                                       */
/*      DELTA('I') = 2                                                       */
/*      DELTA('W') = 3                                                       */
/*      DELTA('O') = 4                                                       */
/*      DELTA('R') = 5                                                       */
/*                                                                           */
/*    This speeds up the string search by jumping over large sections of the */
/*    comparison string.                                                     */
/*                                                                           */
/* 2) There are some further advances we can make in improving the speed     */
/*    of the search.  Consider the situation when we were comparing          */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*      INGROWING                                                            */
/*                                                                           */
/*    we jumped forward by two characters here to align the 'R'.  We could   */
/*    have jumped further if we'd used the additional information due to     */
/*    the fact that we've matched "ING" already.  This means that we could   */
/*    move to align the second "ING" string                                  */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*            INGROWING                                                      */
/*                                                                           */
/*    This is because we know that there is a string "ING" in the searched   */
/*    text and if it did not match the end three characters, it will not     */
/*    match before the "ING" at the beginning.  The pay-off could have       */
/*    been even bigger if the text had been                                  */
/*                                                                           */
/*    DOES STRUNG SEARCHING WORK                                             */
/*      INGROWING                                                            */
/*                                                                           */
/*    Here, we could have skipped to                                         */
/*                                                                           */
/*    DOES STRUNG SEARCHING WORK                                             */
/*               INGROWING                                                   */
/*                                                                           */
/*    as we know that there are no sequences in the pattern that could       */
/*    match "NG" without an 'I' in front.                                    */
/*                                                                           */
/*    The jump table can be calculated as follows                            */
/*                                                                           */
/*    For a matching sequence SUBPAT, we can jump                            */
/*                                                                           */
/*    1) the length of the pattern if SUBPAT does not occur in the string    */
/*       again.                                                              */
/*                                                                           */
/*    2) the offset of the next LEFT most occurance of the pattern, as long  */
/*       as that pattern is preceeded by a difference character to the       */
/*       character to the left of SUBPAT.                                    */
/*                                                                           */
/*    3) the offset so that there would be match with the beginning of the   */
/*       pattern, e.g. the matching sub pattern "WING" match up to the       */
/*       beginning of the pattern "INGROWING" as the last three characters   */
/*       are the same.                                                       */
/*                                                                           */
/*    This option does slow down the generation of the initial table and     */
/*    is sometimes not worth the extra effort to generate the table.  In     */
/*    view of this, a compile flag DELTA2_SKIP can be set to 0 so that the   */
/*    second level skip is not compiled.                                     */
/*                                                                           */
/*    For completeness, a simple example of the standard way of searching a  */
/*    block of text is provided.  This can be enabled by defining            */
/*    SHORT_SEARCH.  This will use the standard 'brute-force' algorithm      */
/*    that is often faster for small strings.                                */
/*****************************************************************************/
/*****************************************************************************/
/* Simple search algorithm                                                   */
/*                                                                           */
/* This algorithm is the simple approach to searching a string.  It uses     */
/* the built in library functions to take advantage of any optimisations     */
/* that have been included.  For small strings and small patterns, this is   */
/* a good choice of algorithm.                                               */
/*                                                                           */
/* It is enabled by compiling with the -DSHORT_SEARCH=1 flag                 */
/*                                                                           */
/* Consider                                                                  */
/*    DOES STRING SEARCHING WORK                                             */
/*    INGROWING                                                              */
/*                                                                           */
/* We search down the text string for an 'I' and compare from there          */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*            INGROWING                                                      */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*                      INGROWING                                            */
/*                                                                           */
/*    DOES STRING SEARCHING WORK                                             */
/*                              INGROWING                                    */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************/
/* Radix search algorithm                                                    */
/*                                                                           */
/* If the string is composed of a small number of characters, the above      */
/* algorithm can be slowed down to a crawl.  This is because the first       */
/* character is continually matching and so we have to perform a string      */
/* comparison of all the characters.  In the event of the input being in     */
/* the form, using a radix search can be an alternative.                     */
/*                                                                           */
/* This can be selected by compiling with -DRADIX_SEARCH=1                   */
/*                                                                           */
/* It works using a similar technique to hashing.  By precalculating the     */
/* hash value of the pattern, it is possible to compare the whole pattern    */
/* with an area of the text string in one go.  Unfortunately, calculating    */
/* the hash pattern of the text string would normally take more work than    */
/* a character by character comparison.  However, by careful choice of hash  */
/* function, the hash can be calculated from the hash value for the previous */
/* text.  Consider a hash function that stores the first 4 characters of     */
/* the string in a 4 byte value, i.e. the hash value for INGROWING is 'INGR' */
/*                                                                           */
/* 1. Calculate the hash value for the start of the text                     */
/* 2. Compare the pattern hash value with the start of the text              */
/* 3. If match, exit                                                         */
/* 4. Calculate the hash of the text starting at the second character, by    */
/*    deriving it from the first.  In this case, we can calculate the hash   */
/*    of the text by subtracting the contents of the hash due to the first   */
/*    character and adding in the fifth character.., i.e. 'DOES' -> 'OES '   */
/*                                                                           */
/* Thus, searching DOES STRING SEARCHING WORK yeilds the following hash value*/
/* comparisons                                                               */
/*                                                                           */
/* 'INGR' == 'DOES'                                                          */
/*           'OES '                                                          */
/*           'ES S'                                                          */
/*           'S ST'                                                          */
/*                                                                           */
/* etc.  In the real life algorithm, a better hash function is chosen which  */
/* takes more characters in to account.  This is even less likely to yeild   */
/* an exact match, so no full string comparison is required.                 */
/*****************************************************************************/
#define INCL_STRING
#define INCL_STDIO
#define INCL_STDLIB
#define INCL_STDDEF
#include "libibm.h"
#include <ctype.h>
#include "srchstr.h"

#if defined(DMALLOC)
#include "dmalloc.h"
#endif

/*****************************************************************************/
/* The maximum character permitted in the search string is 255               */
/* If this is reduced, it will lead to less memory being required by the     */
/* routine, but could cause segmentation violations later.                   */
/*****************************************************************************/
#if !defined(MAX_CHAR)
#define MAX_CHAR (256)
#endif

/*****************************************************************************/
/* Enable or disable the increased skipping capabilities if there is a       */
/* substring that matches.  See algorithm description for more details       */
/*****************************************************************************/
#if !defined(DELTA2_SKIP)
#define DELTA2_SKIP (1)
#endif

#if !defined(SHORT_SEARCH)
#define SHORT_SEARCH (0)
#endif

#if SHORT_SEARCH==0 && RADIX_SEARCH==0
/*****************************************************************************/
/* Procedure    srh_compile                                                  */
/*                                                                           */
/* Purpose      Compile a pattern to use in srh_search                       */
/*                                                                           */
/* Parameters                                                                */
/*              pattern         A block of memory to be used to repeatedly   */
/*                              match to a large section of text             */
/*                                                                           */
/*              pat_len         The length of the data stored at pattern     */
/*                                                                           */
/* Return Code                                                               */
/*              !=NULL          The handle to refer to the compiled pattern  */
/*                              in future.  This should be passed as the     */
/*                              1st parameter to srh_search.  It must be     */
/*                              destroyed using srh_destroy.                 */
/*                                                                           */
/*              NULL            There was not enough memory to satisfy the   */
/*                              request.                                     */
/*****************************************************************************/
#if defined(_NO_PROTO)
srh_prog_t *srh_compile(pattern, pat_len)
byte_t *pattern;
size_t pat_len;
#else
srh_prog_t *srh_compile(byte_t *pattern, size_t pat_len)
#endif
{
  srh_prog_t *prog;
#if DELTA2_SKIP
  size_t subpat_len;
#endif
  unsigned j;

  /***************************************************************************/
  /* Claim space for the pre-search values.  This will hold the delta1 and   */
  /* delta2 search tables, if required.                                      */
  /***************************************************************************/
  prog=(srh_prog_t *)malloc(sizeof(srh_prog_t));
  if (prog!=NULL)
  {
    prog->pattern=pattern;
    prog->pattern_length=pat_len;
    prog->delta1=(size_t *)malloc(sizeof(int)*MAX_CHAR);
    prog->lower_pattern=NULL;
#if DELTA2_SKIP
    prog->delta2=(size_t *)malloc((pat_len+1)*sizeof(int));
#else
    prog->delta2=NULL;
#endif
    if (prog->delta1==NULL
#if DELTA2_SKIP
      || prog->delta2==NULL
#endif
      )
    {
      srh_destroy(prog);
      prog=NULL;
    }
    else
    {
      /***********************************************************************/
      /* Initialize the jump table. Each letter is mapped to its distance    */
      /* the end of the string.  The lastchar, which would be 0, is mapped   */
      /* to the preceeding instance if one exists, or is treated as unfound. */
      /* Unfound characters are mapped to the string's length.               */
      /***********************************************************************/
      for (j=0;j<MAX_CHAR;++j)
        prog->delta1[j]=pat_len;                  /* Default is  target len  */

      for (j=1;j<pat_len;++j)
      {                                           /* Scan target, set table  */
        prog->delta1[(unsigned) pattern[j-1]] = pat_len-j;
      }

#if DELTA2_SKIP
      /***********************************************************************/
      /* By default, the second stage jumps skip the whole string.           */
      /***********************************************************************/
      for (subpat_len=1;subpat_len<pat_len;++subpat_len)
        prog->delta2[pat_len-subpat_len-1]=pat_len+subpat_len;

      /***********************************************************************/
      /* Is there a length of string at the beginning and at the end that is */
      /* the same ? If so, there is a chance that the strings could match    */
      /* earlier .. e.g. a match on "WING" should not move the whole         */
      /* pattern length as if might be possible to match on the "ING" with   */
      /* "INGROWING"                                                         */
      /***********************************************************************/
      for (j=1;j<pat_len;++j)
      {
        if (strncmp((char *)pattern, (char *)pattern+pat_len-j, j)==0)
          for (subpat_len=j; subpat_len<pat_len;++subpat_len)
            prog->delta2[pat_len-subpat_len-1]=pat_len-j+subpat_len;
      }

      /***********************************************************************/
      /* Could the SUBPAT of length subpat_len match completely earlier in   */
      /* the string and the preceeding character does not match the character*/
      /* before the newly matched sub-pattern                                */
      /***********************************************************************/
      for (subpat_len=1; subpat_len<pat_len;++subpat_len)
      {
        for (j=1;j<pat_len;++j)
        {
          if (strncmp((char *)pattern+j,
                      (char *)pattern+pat_len-subpat_len,
                      subpat_len)==0 &&
            pattern[j-1]!=pattern[pat_len-subpat_len-1])
          {
            prog->delta2[pat_len-subpat_len-1]=pat_len-j;
          }
        }
      }

      if (pat_len!=0)
        prog->delta2[pat_len-1]=1;                /* Move 1 if match last!   */

#endif
    }
  }
  return prog;
}

/*****************************************************************************/
/* Procedure    srhi_compile                                                 */
/*                                                                           */
/* Purpose      Compile a pattern to use in srhi_search for case insensitive */
/*              searching.                                                   */
/*                                                                           */
/* Parameters   See srh_compile                                              */
/*                                                                           */
/* Return Code  See srh_compile                                              */
/*****************************************************************************/
#if defined(_NO_PROTO)
srh_prog_t *srhi_compile(pattern, pat_len)
byte_t *pattern;
size_t pat_len;
#else
srh_prog_t *srhi_compile(byte_t *pattern, size_t pat_len)
#endif
{
  srh_prog_t *prog;
#if DELTA2_SKIP
  size_t subpat_len;
#endif
  unsigned j;
  byte_t *lower_pattern;

  prog=(srh_prog_t *)malloc(sizeof(srh_prog_t));
  if (prog!=NULL)
  {
    prog->pattern=pattern;
    prog->pattern_length=pat_len;
    lower_pattern=(byte_t *)malloc((unsigned)pat_len+1);
    prog->lower_pattern=lower_pattern;
    prog->delta1=(size_t *)malloc(sizeof(int)*MAX_CHAR);
#if DELTA2_SKIP
    prog->delta2=(size_t *)malloc((pat_len+1)*sizeof(int));
#else
    prog->delta2=NULL;
#endif
    if (prog->delta1==NULL
#if DELTA2_SKIP
        || prog->delta2==NULL
#endif
        || lower_pattern==NULL)
    {
      srhi_destroy(prog);
      prog=NULL;
    }
    else
    {
      for (j=0;j<pat_len;++j)
        lower_pattern[j]=(char)tolower((int)pattern[j]);
      /***********************************************************************/
      /* Initialize the jump table. Each letter is mapped to its distance    */
      /* the end of the string.  The lastchar, which would be 0, is mapped   */
      /* to the preceeding instance if one exists, or is treated as unfound. */
      /* Unfound characters are mapped to the string's length.               */
      /***********************************************************************/
      for (j=0;j<MAX_CHAR;++j)
        prog->delta1[j]=pat_len;                  /* Default is  target len  */

      for (j=1;j<pat_len;++j)
      {                                           /* Scan target, set table  */
        prog->delta1[(unsigned) lower_pattern[j-1]] = pat_len-j;
        prog->delta1[toupper((int)lower_pattern[j-1])] = pat_len-j;
      }

#if DELTA2_SKIP
      /***********************************************************************/
      /* By default, the second stage jumps skip the whole string.           */
      /***********************************************************************/
      for (subpat_len=1;subpat_len<pat_len;++subpat_len)
        prog->delta2[pat_len-subpat_len-1]=pat_len+subpat_len;

      /***********************************************************************/
      /* Is there a length of string at the beginning and at the end that is */
      /* the same ? If so, there is a chance that the strings could match    */
      /* earlier .. e.g. a match on "WING" should not move the whole         */
      /* pattern length as if might be possible to match on the "ING" with   */
      /* "INGROWING"                                                         */
      /***********************************************************************/
      for (j=1;j<pat_len;++j)
      {
        if (strncmp((char *)lower_pattern, (char *)lower_pattern+pat_len-j, j)==0)
          for (subpat_len=j; subpat_len<pat_len;++subpat_len)
            prog->delta2[pat_len-subpat_len-1]=pat_len-j+subpat_len;
      }

      /***********************************************************************/
      /* Could the SUBPAT of length subpat_len match completely earlier in   */
      /* the string and the preceeding character does not match the character*/
      /* before the newly matched sub-pattern                                */
      /***********************************************************************/
      for (subpat_len=1; subpat_len<pat_len;++subpat_len)
      {
        for (j=1;j<pat_len;++j)
        {
          if (strncmp((char *)lower_pattern+j,
                      (char *)lower_pattern+pat_len-subpat_len,
                      subpat_len)==0 &&
            lower_pattern[j-1]!=lower_pattern[pat_len-subpat_len-1])
          {
            prog->delta2[pat_len-subpat_len-1]=pat_len-j;
          }
        }
      }

      if (pat_len!=0)
        prog->delta2[pat_len-1]=1;                /* Move 1 if match last!   */
#endif
    }
  }
  return prog;
}

/*****************************************************************************/
/* Procedure    srh_search                                                   */
/*                                                                           */
/* Purpose      Search the text for a pattern previously compiled with       */
/*              srh_compile                                                  */
/*                                                                           */
/* Parameters                                                                */
/*              text            A block of text to be searched for the       */
/*                              pattern.                                     */
/*                                                                           */
/*              text_len        The length of the data stored at text        */
/*                                                                           */
/* Return Code                                                               */
/*              !=-1            There was a match of the pattern at this     */
/*                              offset in to the string.                     */
/*                                                                           */
/*              -1              There was no pattern in the text passed      */
/*****************************************************************************/
#if defined(_NO_PROTO)
int srh_search(prog, text, text_len)
srh_prog_t *prog;
byte_t *text;
size_t text_len;
#else
int srh_search(srh_prog_t *prog, byte_t *text, size_t text_len)
#endif
{
  size_t pat_len;                                 /* Length of pattern       */
  size_t lastpos;                                 /* Offset of last char in  */
                                                  /*  pattern                */
  byte_t lastchar;                                /* The last char in pattern*/
  byte_t *pattern;                                /* The pattern             */
  size_t  i;
  size_t  j;
  size_t *delta1;
#if DELTA2_SKIP
  size_t delta1_value;
  size_t *delta2;
  size_t delta2_value;
#else
  size_t delta1_lastchar;
  size_t back_i;
#endif

  pat_len=prog->pattern_length;
  if (pat_len==0)
    return 0;

  pattern=prog->pattern;
  lastpos=pat_len-1;                              /* Offset to last char     */
  lastchar=pattern[lastpos];                      /* Quick access to last    */
  delta1=prog->delta1;
#if DELTA2_SKIP
  delta2=prog->delta2;
#else
  delta1_lastchar=delta1[lastchar];
#endif

  i=lastpos;                                      /* Start at first possible */
  while(i<text_len)                               /* Run out of input ?      */
  {
    /*************************************************************************/
    /* Quickly skip over all possible text till we get a match on the last   */
    /* character or we run out of input                                      */
    /*************************************************************************/
    if (text[i]==lastchar)
    {
#if DELTA2_SKIP
      j=lastpos;
      do
      {
        if (j--==0)
          return i;
        --i;
      } while (text[i]==pattern[j]);
      delta1_value=delta1[text[i]];
      delta2_value=delta2[j];
      if (delta1_value>delta2_value)
        i+=delta1_value;
      else
        i+=delta2_value;
#else
      if (lastpos==0)
        return i;
      j=lastpos-1;
      back_i=i-1;
      while (text[back_i]==pattern[j])
      {
        if (j--==0)
          return back_i;
        --back_i;
      }
      i+=delta1_lastchar;
#endif
    }
    else
      i+=delta1[text[i]];
  }
  return -1;
}

/*****************************************************************************/
/* Procedure    srhi_search                                                  */
/*                                                                           */
/* Purpose      Search the text for a pattern previously compiled with       */
/*              srhi_compile, i.e. with case insensitive                     */
/*                                                                           */
/* Parameters   See srh_search                                               */
/*                                                                           */
/* Return Code  See srh_search                                               */
/*****************************************************************************/
#if defined(_NO_PROTO)
int srhi_search(prog, text, text_len)
srh_prog_t *prog;
byte_t *text;
size_t text_len;
#else
int srhi_search(srh_prog_t *prog, byte_t *text, size_t text_len)
#endif
{
  size_t pat_len;                                 /* Length of pattern       */
  size_t lastpos;                                 /* Offset of last char in  */
                                                  /*  pattern                */
  byte_t lastchar;                                /* The last char in pattern*/
  byte_t upperchar;                               /* Upper case version of   */
                                                  /*  lastchar               */
  byte_t *pattern;                                /* The pattern             */
  size_t  i;
  size_t  j;
  size_t *delta1;
#if DELTA2_SKIP
  size_t *delta2;
  size_t delta1_value;
  size_t delta2_value;
#else
  size_t delta1_lastchar;
  size_t back_i;
#endif

  pat_len=prog->pattern_length;
  if (pat_len==0)
    return 0;

  pattern=prog->lower_pattern;
  delta1=prog->delta1;
  lastpos=pat_len-1;                              /* Offset to last char     */
  lastchar=pattern[lastpos];                      /* Quick access to last    */
  upperchar=(char)toupper((int)lastchar);         /* Upper case copy of last */
#if DELTA2_SKIP
  delta2=prog->delta2;
#else
  delta1_lastchar=delta1[lastchar];
#endif

  i=lastpos;                                      /* Start at first possible */
  while(i<text_len)                               /* Run out of input ?      */
  {
    /*************************************************************************/
    /* Quickly skip over all possible text till we get a match on the last   */
    /* character or we run out of input                                      */
    /*************************************************************************/
    if (text[i]==lastchar || text[i]==upperchar)
    {
#if DELTA2_SKIP
      j=lastpos;
      do
      {
        if (j--==0)
          return i;
        --i;
      } while ((byte_t)tolower((int)text[i])==pattern[j]);
      delta1_value=delta1[text[i]];
      delta2_value=delta2[j];
      if (delta1_value>delta2_value)
        i+=delta1_value;
      else
        i+=delta2_value;
#else
      if (lastpos==0)
        return i;
      j=lastpos-1;
      back_i=i-1;
      while ((byte_t)tolower((int)text[back_i])==pattern[j])
      {
        if (j--==0)
          return back_i;
        --back_i;
      }
      i+=delta1_lastchar;
#endif
    }
    else
      i+=delta1[text[i]];
  }
  return -1;
}

/*****************************************************************************/
/* Procedure    srh_destroy                                                  */
/*                                                                           */
/* Purpose      Frees up all resources associated with the pattern.          */
/*                                                                           */
/* Parameters                                                                */
/*              prog            The result of a srh_compile call.            */
/*****************************************************************************/
#if defined(_NO_PROTO)
void srh_destroy(prog)
srh_prog_t *prog;
#else
void srh_destroy(srh_prog_t *prog)
#endif

{
  if (prog!=NULL)
  {
    if (prog->delta1!=NULL)
      free((void_ptr_t)prog->delta1);
#if DELTA2_SKIP
    if (prog->delta2!=NULL)
      free((void_ptr_t)prog->delta2);
#endif
    if (prog->lower_pattern!=NULL)
      free((void_ptr_t)prog->lower_pattern);
    prog->pattern=NULL;
    prog->pattern_length=0;
    prog->delta1=NULL;
    prog->delta2=NULL;
    prog->lower_pattern=NULL;
    free((void_ptr_t)prog);
  }
  return;
}

/*****************************************************************************/
/* Procedure    srhi_destroy                                                 */
/*                                                                           */
/* Purpose      Frees up all resources associated with case insensitive      */
/*              pattern                                                      */
/*                                                                           */
/* Parameters                                                                */
/*              prog            The result of a srhi_compile call.           */
/*****************************************************************************/
#if defined(_NO_PROTO)
void srhi_destroy(prog)
srh_prog_t *prog;
#else
void srhi_destroy(srh_prog_t *prog)
#endif
{
  srh_destroy(prog);
  return;
}

/*****************************************************************************/
/* Procedure    strsrh                                                       */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string                       */
/*                                                                           */
/* Parameters                                                                */
/*              text            A block of text to be searched for the       */
/*                              pattern.  This is a null terminated string.  */
/*                                                                           */
/*              pattern         The pattern to be searched for.              */
/*                              This is a null terminated string             */
/*                                                                           */
/* Return Code                                                               */
/*              !=-1            There was a match of the pattern at this     */
/*                              offset in to the text.                       */
/*                                                                           */
/*              -1              There was no pattern in the text passed      */
/*                                                                           */
/* Note         If a pattern is going to be searched for more than once, it  */
/*              is recommended to use the srh_compile, srh_search,           */
/*              srh_destroy calls instead.  This will reduce the overhead of */
/*              compiling the pattern.                                       */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrh(text, pattern)
char *text;
char *pattern;
#else
int strsrh(char *text, char *pattern)
#endif
{
  srh_prog_t *srh_prog;
  int match_offset;

  srh_prog=srh_compile((byte_t *)pattern,strlen(pattern));
  if (srh_prog!=NULL)
  {
    match_offset=srh_search(srh_prog, (byte_t *)text, strlen(text));
  }
  else
    match_offset= -1;

  srh_destroy(srh_prog);
  return match_offset;
}

/*****************************************************************************/
/* Procedure    strsrhi                                                      */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string (case insensitive)    */
/*                                                                           */
/* Parameters   See strsrh                                                   */
/*                                                                           */
/* Return Code  See strsrh                                                   */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrhi(text, pattern)
char *text;
char *pattern;
#else
int strsrhi(char *text, char *pattern)
#endif
{
  srh_prog_t *srh_prog;
  int match_offset;

  srh_prog=srhi_compile((byte_t *)pattern,strlen(pattern));
  if (srh_prog!=NULL)
  {
    match_offset=srhi_search(srh_prog, (byte_t *)text, strlen(text));
  }
  else
    match_offset= -1;

  srhi_destroy(srh_prog);
  return match_offset;
}
#else

/*****************************************************************************/
/* AIX/RT does not have an ignore-case string comparison routine             */
/* Neither does the S/6000 ...                                               */
/*****************************************************************************/
#if defined(CC_PCC) || defined(CC_XLC) || defined(CC_GNU)
#if defined(_NO_PROTO)
static int strnicmp(str1, str2, count)
char *str1;
char *str2;
size_t count;
#else
static int strnicmp(char *str1, char *str2, size_t count)
#endif
{
  while (*str1!='\0' && *str2!='\0' && count!=0)
  {
    if (toupper(*str1)!=toupper(*str2))
      break;
    ++str1;
    ++str2;
    --count;
  }
  if (count==0)
    return 0;

  return *str1-*str2;
}
#endif


#if SHORT_SEARCH
/*****************************************************************************/
/* Procedure    strsrh                                                       */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string                       */
/*                                                                           */
/* Parameters                                                                */
/*              text            A block of text to be searched for the       */
/*                              pattern.  This is a null terminated string.  */
/*                                                                           */
/*              pattern         The pattern to be searched for.              */
/*                              This is a null terminated string             */
/*                                                                           */
/* Return Code                                                               */
/*              !=-1            There was a match of the pattern at this     */
/*                              offset in to the text.                       */
/*                                                                           */
/*              -1              There was no pattern in the text passed      */
/*                                                                           */
/* Note         The simple approach to string searching.  This is often      */
/*              faster than more complex methods which pre-process the       */
/*              pattern.                                                     */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrh(text, pattern)
char *text;
char *pattern;
#else
int strsrh(char *text, char *pattern)
#endif
{
  int first_char=(int)*pattern;
  char *current_text_position=text;
  size_t pattern_length;

  if (first_char=='\0')
    return 0;

  ++pattern;
  pattern_length=strlen(pattern);

  current_text_position=strchr(current_text_position, first_char);
  while (current_text_position!=NULL)
  {
    if (strncmp(current_text_position+1, pattern, pattern_length)==0)
      return current_text_position-text;
    current_text_position=strchr(current_text_position+1, first_char);
  }
  return -1;
}

/*****************************************************************************/
/* Procedure    strsrhi                                                      */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string (case insensitive)    */
/*                                                                           */
/* Parameters   See strsrh                                                   */
/*                                                                           */
/* Return Code  See strsrh                                                   */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrhi(text, pattern)
char *text;
char *pattern;
#else
int strsrhi(char *text, char *pattern)
#endif
{
  int upper_first_char=toupper(*pattern);
  int lower_first_char=tolower(*pattern);
  char *upper_text_position;
  char *lower_text_position;
  size_t pattern_length;

  if (*pattern=='\0')
    return 0;

  ++pattern;
  pattern_length=strlen(pattern);

  upper_text_position=strchr(text, upper_first_char);
  lower_text_position=strchr(text, lower_first_char);
  if (upper_first_char==lower_first_char)
  {
    while (upper_text_position!=NULL)
    {
      if (strnicmp(upper_text_position+1, pattern, pattern_length)==0)
        return upper_text_position-text;
      upper_text_position=strchr(upper_text_position+1, upper_first_char);
    }
  }
  else
  {
    while (upper_text_position!=NULL && lower_text_position!=NULL)
    {
      if (lower_text_position<upper_text_position)
      {
        if (strnicmp(lower_text_position+1, pattern, pattern_length)==0)
          return lower_text_position-text;
        lower_text_position=strchr(lower_text_position+1, lower_first_char);
      }
      else
      {
        if (strnicmp(upper_text_position+1, pattern, pattern_length)==0)
          return upper_text_position-text;
        upper_text_position=strchr(upper_text_position+1, upper_first_char);
      }
    }

    if (lower_text_position==NULL)
    {
      while (upper_text_position!=NULL)
      {
        if (strnicmp(upper_text_position+1, pattern, pattern_length)==0)
          return upper_text_position-text;
        upper_text_position=strchr(upper_text_position+1, upper_first_char);
      }
    }
    else
    {
      while (lower_text_position!=NULL)
      {
        if (strnicmp(lower_text_position+1, pattern, pattern_length)==0)
          return lower_text_position-text;
        lower_text_position=strchr(lower_text_position+1, lower_first_char);
      }
    }
  }
  return -1;
}

#else

#define HASH_INCLUDE(hash, value) (hash)=((hash)<<1)^(value)
#define HASH_REMOVE(hash, value)  (hash)=(hash)^((value)<<((pattern_length)-1))

/*****************************************************************************/
/* Procedure    strsrh                                                       */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string                       */
/*                                                                           */
/* Parameters                                                                */
/*              text            A block of text to be searched for the       */
/*                              pattern.  This is a null terminated string.  */
/*                                                                           */
/*              pattern         The pattern to be searched for.              */
/*                              This is a null terminated string             */
/*                                                                           */
/* Return Code                                                               */
/*              !=-1            There was a match of the pattern at this     */
/*                              offset in to the text.                       */
/*                                                                           */
/*              -1              There was no pattern in the text passed      */
/*                                                                           */
/* Note         Uses the radix algorithm to speed up searching with          */
/*              repetative patterns                                          */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrh(text, pattern)
char *text;
char *pattern;
#else
int strsrh(char *text, char *pattern)
#endif
{
  unsigned i;
  unsigned pattern_length;
  unsigned pattern_hash=0;
  unsigned text_hash=0;
  char *end_of_match_ptr;
  char *beg_of_match_ptr;

  pattern_length=strlen(pattern);
  if (pattern_length==0)
    return 0;

  /***************************************************************************/
  /* Calculate the hash value of the pattern                                 */
  /***************************************************************************/
  for (i=0; i<pattern_length; ++i)
  {
    HASH_INCLUDE(pattern_hash, pattern[i]);
  }

  /***************************************************************************/
  /* Calculate the hash value of the start of the text.  If the text is      */
  /* shorter than the pattern, it cannot match                               */
  /***************************************************************************/
  for (i=0;i<pattern_length; ++i)
  {
    HASH_INCLUDE(text_hash, text[i]);
    if (text[i]=='\0')
      return -1;
  }

  /***************************************************************************/
  /* Now try and match the string                                            */
  /***************************************************************************/
  beg_of_match_ptr=text;
  end_of_match_ptr=text+i-1;
  while (*end_of_match_ptr!='\0')
  {
    if (pattern_hash==text_hash)                  /* Match the hashes ?      */
    {
      if (strncmp(beg_of_match_ptr, pattern, pattern_length)==0)/* Check     */
      {
        return beg_of_match_ptr-text;
      }
    }
    /*************************************************************************/
    /* Remove the results of the first character comparison and add in the   */
    /* next character in the text                                            */
    /*************************************************************************/
    HASH_REMOVE(text_hash, *beg_of_match_ptr);
    ++beg_of_match_ptr;
    ++end_of_match_ptr;
    HASH_INCLUDE(text_hash, *end_of_match_ptr);
  }
  return -1;
}

#define UHASH_INCLUDE(hash, value) (HASH_INCLUDE(hash, (toupper(value))))
#define UHASH_REMOVE(hash, value) (HASH_REMOVE(hash, (toupper(value))))
/*****************************************************************************/
/* Procedure    strsrhi                                                      */
/*                                                                           */
/* Purpose      Tests if a pattern is in a text string (case insensitive)    */
/*                                                                           */
/* Parameters   See strsrh                                                   */
/*                                                                           */
/* Return Code  See strsrh                                                   */
/*****************************************************************************/
#if defined(_NO_PROTO)
int strsrhi(text, pattern)
char *text;
char *pattern;
#else
int strsrhi(char *text, char *pattern)
#endif
{
  unsigned i;
  unsigned pattern_length;
  unsigned pattern_hash=0;
  unsigned text_hash=0;
  char *end_of_match_ptr;
  char *beg_of_match_ptr;

  pattern_length=strlen(pattern);
  if (pattern_length==0)
    return 0;

  for (i=0; i<pattern_length; ++i)
  {
    UHASH_INCLUDE(pattern_hash, pattern[i]);
  }
  for (i=0;i<pattern_length; ++i)
  {
    UHASH_INCLUDE(text_hash, text[i]);
    if (text[i]=='\0')
      return -1;
  }
  beg_of_match_ptr=text;
  end_of_match_ptr=text+i-1;
  while (*end_of_match_ptr!='\0')
  {
    if (pattern_hash==text_hash)
    {
      if (strnicmp(beg_of_match_ptr, pattern, pattern_length)==0)
      {
        return beg_of_match_ptr-text;
      }
    }
    UHASH_REMOVE(text_hash, *beg_of_match_ptr);
    ++beg_of_match_ptr;
    ++end_of_match_ptr;
    UHASH_INCLUDE(text_hash, *end_of_match_ptr);
  };
  return -1;
}
#endif
#endif

#if defined(TEST)
#include "libibm.h"

/*****************************************************************************/
/* The search test cases                                                     */
/*                                                                           */
/* They specify the text and pattern, followed by the expected return codes  */
/* for case sensitive and insensitive search                                 */
/*****************************************************************************/
struct
{
  char *haystack;
  char *needle;
  int   offset;                                           /* Rc from strsrch */
  int  ioffset;                                           /*         strsrchi*/
} data[] = {
  { "Test", "e",     1,1  },
  { "Test", "s",     2,2  },
  { "test", "test", 0,0  },
  { "atest", "test", 1,1  },
  { "aatest", "atest", 1,1  },
  { "test", "test string", -1, -1 },
  { "this is a longer test string","test",17,17 },
  { "this is one should not match","test",-1,-1 },
  { "an even longer string that should allow the comparison to really speed\
along until it finally finds the needle it is looking for","needle",103,103 },
  { "Test", "test", -1,0  },
  { "Test", "tesT", -1,0  },
  { "Test", "teST", -1,0  },
  { "Test", "tEst", -1,0  },
  { "Test", "TeSt", -1,0  },
  { "TEST", "TeSt", -1,0  },
  { "TEST", "TeST", -1,0  },
  { "TEST", "TESt", -1,0  },
  { "",     "",      0,0  },
  { "Test", "",      0,0  },
  { "",     "Test",  -1,-1},
  { "Testing", "Ti", -1,3 },
  { "TestIng", "T@I", -1,-1 },
  { "TestIng", "An enormous example of a search string T@", -1,-1 },
  { "TestIng", "T@", -1,-1 },
  { "An enormous example of an @", "An enormous example of a search string T@", -1,-1 },
  { "this is a longer Test string","Test",17,17 },
  { "this is a longer test string","Test",-1,17 },
  { "test is a longer Test string","Test",17,0 },
  { "this is one should not Match","test",-1,-1 },
  { "an even longer string that should allow the comparison to really\
speed along until it finally finds the Needle it is looking for",
    "NeeDlE",-1,103 },
  { "1aaa2aaa3aaa4aa5a6aaa7aaaa8aaaa9aa0aaaaa","aaaaaa",-1,-1 },
  { "1aaa2aaa3aaa4aa5a6aaa7aaaa8aaaa9aa0aaaaaa","aaaaaa",35,35 },
  { "1234567890123456789", "678", 5, 5 },
  { "0000011111222223333", "01", 4, 4 },
  { "0000011111222223333", "34", -1, -1 },
  { "MatchingMutchingMitchingStrings","Mitching", 16,16 },
};

#if defined(_NO_PROTO)
int main(argc,argv)
int argc;
char *argv[];
#else
int main(int argc, char *argv[])
#endif
{
  int i;
  int rc;
  int prog_rc=0;

  if (argc<3)
  {
    for (i=0;i<COUNTOF(data);++i)
    {
      rc=strsrh(data[i].haystack,data[i].needle);
      if (rc!=data[i].offset)
      {
        (void)printf("ERROR: strsrh test %d rc=%d data.offset=%d \"%s\" \"%s\"\n",i,rc,data[i].offset,data[i].haystack, data[i].needle);
        prog_rc=1;
      }
      rc=strsrhi(data[i].haystack, data[i].needle);
      if (rc!=data[i].ioffset)
      {
        (void)printf("ERROR: strsrhi test %d rc=%d data.offset=%d \"%s\" \"%s\"\n",i,rc,data[i].ioffset,data[i].haystack, data[i].needle);
        prog_rc=1;
      }
    }
  }
  else
  {
    rc=strsrhi(argv[2],argv[1]);
    if (rc==-1)
    {
      (void)printf("ERROR: could not find (ignore case) \"%s\" in \"%s\"\n", argv[1], argv[2]);
      prog_rc=1;
    }
    rc=strsrh(argv[2],argv[1]);
    if (rc==-1)
    {
      (void)printf("ERROR: could not find \"%s\" in \"%s\"\n", argv[1], argv[2]);
      prog_rc=1;
    }
  }
#if defined(DMALLOC)
  ddump();
#endif
  return prog_rc;
}
#endif


